#include "alert.h"
#include "anomaly_detector.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <syslog.h>
#include <stdarg.h>

static AlertConfig *g_alert_config = NULL;

int alert_init(AlertConfig *config) {
    if (!config) {
        return -1;
    }

    g_alert_config = config;

    if (config->syslog_enabled) {
        openlog("pwarden", LOG_PID | LOG_CONS, LOG_DAEMON);
    }

    return 0;
}

void alert_cleanup(void) {
    if (g_alert_config && g_alert_config->syslog_enabled) {
        closelog();
    }
    g_alert_config = NULL;
}

static void format_timestamp(time_t timestamp, char *buffer, size_t size) {
    struct tm *tm_info = localtime(&timestamp);
    strftime(buffer, size, "%Y-%m-%d %H:%M:%S", tm_info);
}

int alert_send_to_stdout(Anomaly *anomaly) {
    if (!anomaly) {
        return -1;
    }

    char timestamp[64];
    format_timestamp(anomaly->timestamp, timestamp, sizeof(timestamp));

    printf("[%s] ANOMALY DETECTED\n", timestamp);
    printf("  Type: %s\n", anomaly_type_to_string(anomaly->type));
    printf("  PID: %d\n", anomaly->pid);
    printf("  Process: %s\n", anomaly->process_name);
    printf("  Severity: %.2f\n", anomaly->severity);
    printf("  Details: %s\n", anomaly->details);
    printf("\n");

    return 0;
}

int alert_send_to_logfile(Anomaly *anomaly, const char *logfile_path) {
    if (!anomaly || !logfile_path) {
        return -1;
    }

    FILE *file = fopen(logfile_path, "a");
    if (!file) {
        return -1;
    }

    char timestamp[64];
    format_timestamp(anomaly->timestamp, timestamp, sizeof(timestamp));

    fprintf(file, "[%s] ANOMALY: type=%s pid=%d process=%s severity=%.2f details=\"%s\"\n",
            timestamp, anomaly_type_to_string(anomaly->type), anomaly->pid,
            anomaly->process_name, anomaly->severity, anomaly->details);
    fflush(file);
    fclose(file);

    return 0;
}

int alert_send_to_syslog(Anomaly *anomaly) {
    if (!anomaly) {
        return -1;
    }

    int priority = LOG_WARNING;
    if (anomaly->severity >= 0.8) {
        priority = LOG_ERR;
    } else if (anomaly->severity >= 0.5) {
        priority = LOG_WARNING;
    } else {
        priority = LOG_INFO;
    }

    syslog(priority, "ANOMALY: type=%s pid=%d process=%s severity=%.2f details=\"%s\"",
           anomaly_type_to_string(anomaly->type), anomaly->pid,
           anomaly->process_name, anomaly->severity, anomaly->details);

    return 0;
}

int alert_send(Anomaly *anomaly) {
    if (!anomaly || !g_alert_config) {
        return -1;
    }

    int ret = 0;

    if (g_alert_config->stdout_enabled) {
        ret |= alert_send_to_stdout(anomaly);
    }

    if (g_alert_config->logfile_enabled && g_alert_config->logfile_path) {
        ret |= alert_send_to_logfile(anomaly, g_alert_config->logfile_path);
    }

    if (g_alert_config->syslog_enabled) {
        ret |= alert_send_to_syslog(anomaly);
    }

    return ret;
}

const char* anomaly_type_to_string(AnomalyType type) {
    switch (type) {
        case ANOMALY_NETWORK_UNKNOWN_CONNECTION:
            return "NETWORK_UNKNOWN_CONNECTION";
        case ANOMALY_RESOURCE_CPU_SPIKE:
            return "RESOURCE_CPU_SPIKE";
        case ANOMALY_RESOURCE_MEMORY_SPIKE:
            return "RESOURCE_MEMORY_SPIKE";
        case ANOMALY_FILE_UNUSUAL_ACCESS:
            return "FILE_UNUSUAL_ACCESS";
        case ANOMALY_PROCESS_SPAWN:
            return "PROCESS_SPAWN";
        default:
            return "UNKNOWN";
    }
}

