#include "anomaly_detector.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

static bool network_connection_exists(NetworkBaselineData *baseline, const char *ip, uint16_t port, char protocol) {
    NetworkBaseline *connections;
    size_t count;

    if (protocol == 'T') {
        connections = baseline->tcp_connections;
        count = baseline->tcp_count;
    } else {
        connections = baseline->udp_connections;
        count = baseline->udp_count;
    }

    for (size_t i = 0; i < count; i++) {
        if (connections[i].dest_ip && 
            strcmp(connections[i].dest_ip, ip) == 0 && 
            connections[i].dest_port == port) {
            return true;
        }
    }
    return false;
}

static bool file_matches_pattern(const char *file, const char *pattern) {
    // Simple pattern matching - check if file starts with pattern (minus the /*)
    size_t pattern_len = strlen(pattern);
    if (pattern_len > 0 && pattern[pattern_len - 1] == '*' && pattern[pattern_len - 2] == '/') {
        pattern_len -= 2;
    }
    return strncmp(file, pattern, pattern_len) == 0;
}

int anomaly_detector_check_network(ProcessBaseline *baseline, BehaviorSnapshot *snapshot, Anomaly **anomalies, size_t *count) {
    if (!baseline || !snapshot || !anomalies || !count) {
        return -1;
    }

    *count = 0;
    *anomalies = NULL;
    size_t capacity = 0;

    for (size_t i = 0; i < snapshot->network.count; i++) {
        NetworkConnection *conn = &snapshot->network.connections[i];
        
        if (!network_connection_exists(&baseline->network, conn->dest_ip, conn->dest_port, conn->protocol)) {
            // Unknown connection detected
            if (*count >= capacity) {
                capacity = capacity ? capacity * 2 : 4;
                *anomalies = realloc(*anomalies, capacity * sizeof(Anomaly));
            }

            Anomaly *anom = &(*anomalies)[*count];
            anom->type = ANOMALY_NETWORK_UNKNOWN_CONNECTION;
            anom->pid = snapshot->pid;
            anom->process_name = strdup(baseline->name);
            anom->timestamp = snapshot->timestamp;
            anom->severity = 0.8; // High severity for unknown connections

            char details[256];
            snprintf(details, sizeof(details), "Unknown %c connection to %s:%d", 
                    conn->protocol, conn->dest_ip, conn->dest_port);
            anom->details = strdup(details);
            (*count)++;
        }
    }

    return 0;
}

int anomaly_detector_check_resources(ProcessBaseline *baseline, BehaviorSnapshot *snapshot, double threshold, Anomaly **anomalies, size_t *count) {
    if (!baseline || !snapshot || !anomalies || !count) {
        return -1;
    }

    *count = 0;
    *anomalies = NULL;
    size_t capacity = 0;

    // Check memory usage
    if (baseline->resources.memory_std > 0) {
        double memory_z = ((double)snapshot->resources.memory_rss - baseline->resources.memory_mean) / baseline->resources.memory_std;
        
        if (fabs(memory_z) > threshold) {
            if (*count >= capacity) {
                capacity = capacity ? capacity * 2 : 4;
                *anomalies = realloc(*anomalies, capacity * sizeof(Anomaly));
            }

            Anomaly *anom = &(*anomalies)[*count];
            anom->type = memory_z > 0 ? ANOMALY_RESOURCE_MEMORY_SPIKE : ANOMALY_RESOURCE_MEMORY_SPIKE;
            anom->pid = snapshot->pid;
            anom->process_name = strdup(baseline->name);
            anom->timestamp = snapshot->timestamp;
            anom->severity = fmin(1.0, fabs(memory_z) / (threshold * 2));

            char details[256];
            snprintf(details, sizeof(details), "Memory usage: %lu bytes (mean: %.0f, std: %.0f, z-score: %.2f)",
                    snapshot->resources.memory_rss, baseline->resources.memory_mean, 
                    baseline->resources.memory_std, memory_z);
            anom->details = strdup(details);
            (*count)++;
        }
    }

    // Check CPU usage (if we had proper CPU tracking)
    if (baseline->resources.cpu_std > 0 && snapshot->resources.cpu_percent > 0) {
        double cpu_z = (snapshot->resources.cpu_percent - baseline->resources.cpu_mean) / baseline->resources.cpu_std;
        
        if (cpu_z > threshold) {
            if (*count >= capacity) {
                capacity = capacity ? capacity * 2 : 4;
                *anomalies = realloc(*anomalies, capacity * sizeof(Anomaly));
            }

            Anomaly *anom = &(*anomalies)[*count];
            anom->type = ANOMALY_RESOURCE_CPU_SPIKE;
            anom->pid = snapshot->pid;
            anom->process_name = strdup(baseline->name);
            anom->timestamp = snapshot->timestamp;
            anom->severity = fmin(1.0, cpu_z / (threshold * 2));

            char details[256];
            snprintf(details, sizeof(details), "CPU usage: %.2f%% (mean: %.2f, std: %.2f, z-score: %.2f)",
                    snapshot->resources.cpu_percent, baseline->resources.cpu_mean,
                    baseline->resources.cpu_std, cpu_z);
            anom->details = strdup(details);
            (*count)++;
        }
    }

    return 0;
}

int anomaly_detector_check_files(ProcessBaseline *baseline, BehaviorSnapshot *snapshot, Anomaly **anomalies, size_t *count) {
    if (!baseline || !snapshot || !anomalies || !count) {
        return -1;
    }

    *count = 0;
    *anomalies = NULL;
    size_t capacity = 0;

    for (size_t i = 0; i < snapshot->files.count; i++) {
        const char *file = snapshot->files.files[i];
        bool matches = false;

        // Check if file matches any known pattern
        for (size_t j = 0; j < baseline->files.count; j++) {
            if (file_matches_pattern(file, baseline->files.patterns[j])) {
                matches = true;
                break;
            }
        }

        if (!matches) {
            // Unusual file access
            if (*count >= capacity) {
                capacity = capacity ? capacity * 2 : 4;
                *anomalies = realloc(*anomalies, capacity * sizeof(Anomaly));
            }

            Anomaly *anom = &(*anomalies)[*count];
            anom->type = ANOMALY_FILE_UNUSUAL_ACCESS;
            anom->pid = snapshot->pid;
            anom->process_name = strdup(baseline->name);
            anom->timestamp = snapshot->timestamp;
            anom->severity = 0.6; // Medium severity

            char details[512];
            snprintf(details, sizeof(details), "Unusual file access: %s", file);
            anom->details = strdup(details);
            (*count)++;
        }
    }

    return 0;
}

// Legacy function - kept for compatibility but not used in main
Anomaly* anomaly_detector_check(ProcessBaseline *baseline, BehaviorSnapshot *snapshot, double threshold) {
    // This function is deprecated - use the specific check functions instead
    (void)baseline;
    (void)snapshot;
    (void)threshold;
    return NULL;
}

void anomaly_free(Anomaly *anomaly) {
    if (!anomaly) {
        return;
    }

    free(anomaly->process_name);
    free(anomaly->details);
    free(anomaly);
}

const char* anomaly_type_to_string(AnomalyType type) {
    switch (type) {
        case ANOMALY_NETWORK_UNKNOWN_CONNECTION:
            return "NETWORK_UNKNOWN_CONNECTION";
        case ANOMALY_RESOURCE_CPU_SPIKE:
            return "RESOURCE_CPU_SPIKE";
        case ANOMALY_RESOURCE_MEMORY_SPIKE:
            return "RESOURCE_MEMORY_SPIKE";
        case ANOMALY_FILE_UNUSUAL_ACCESS:
            return "FILE_UNUSUAL_ACCESS";
        case ANOMALY_PROCESS_SPAWN:
            return "PROCESS_SPAWN";
        default:
            return "UNKNOWN";
    }
}

