#include "baseline.h"
#include "json_utils.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <math.h>
#include <time.h>

#define MAX_PATH_LEN 4096

static void network_baseline_add_connection(NetworkBaselineData *data, const char *ip, uint16_t port, char protocol) {
    NetworkBaseline *baselines;
    size_t *count;
    size_t *capacity;

    if (protocol == 'T') {
        baselines = data->tcp_connections;
        count = &data->tcp_count;
        capacity = &data->tcp_capacity;
    } else {
        baselines = data->udp_connections;
        count = &data->udp_count;
        capacity = &data->udp_capacity;
    }

    // Check if connection already exists
    for (size_t i = 0; i < *count; i++) {
        if (baselines[i].dest_ip && strcmp(baselines[i].dest_ip, ip) == 0 && baselines[i].dest_port == port) {
            baselines[i].count++;
            return;
        }
    }

    // Add new connection
    if (*count >= *capacity) {
        *capacity = *capacity ? *capacity * 2 : 16;
        if (protocol == 'T') {
            data->tcp_connections = realloc(data->tcp_connections, *capacity * sizeof(NetworkBaseline));
            baselines = data->tcp_connections;
        } else {
            data->udp_connections = realloc(data->udp_connections, *capacity * sizeof(NetworkBaseline));
            baselines = data->udp_connections;
        }
    }

    baselines[*count].dest_ip = strdup(ip);
    baselines[*count].dest_port = port;
    baselines[*count].count = 1;
    baselines[*count].frequency = 0.0;
    (*count)++;
}

static void file_baseline_add_pattern(FileBaseline *baseline, const char *file) {
    // Simple pattern matching - store directory patterns
    char *pattern = strdup(file);
    if (!pattern) {
        return;
    }

    // Extract directory pattern (simplified)
    char *last_slash = strrchr(pattern, '/');
    if (last_slash) {
        *last_slash = '\0';
        strcat(pattern, "/*");
    }

    // Check if pattern already exists
    for (size_t i = 0; i < baseline->count; i++) {
        if (strcmp(baseline->patterns[i], pattern) == 0) {
            free(pattern);
            return;
        }
    }

    if (baseline->count >= baseline->capacity) {
        baseline->capacity = baseline->capacity ? baseline->capacity * 2 : 16;
        baseline->patterns = realloc(baseline->patterns, baseline->capacity * sizeof(char*));
    }

    baseline->patterns[baseline->count] = pattern;
    baseline->count++;
}

BaselineStore* baseline_init(void) {
    BaselineStore *store = calloc(1, sizeof(BaselineStore));
    store->capacity = 16;
    store->baselines = calloc(store->capacity, sizeof(ProcessBaseline));
    return store;
}

void baseline_destroy(BaselineStore *store) {
    if (!store) {
        return;
    }

    for (size_t i = 0; i < store->count; i++) {
        process_baseline_free(&store->baselines[i]);
    }
    free(store->baselines);
    free(store);
}

ProcessBaseline* baseline_create_new(BaselineStore *store, pid_t pid, const char *name, const char *executable) {
    if (!store) {
        return NULL;
    }

    if (store->count >= store->capacity) {
        store->capacity *= 2;
        store->baselines = realloc(store->baselines, store->capacity * sizeof(ProcessBaseline));
    }

    ProcessBaseline *baseline = &store->baselines[store->count];
    memset(baseline, 0, sizeof(ProcessBaseline));

    baseline->pid = pid;
    baseline->name = strdup(name ? name : "");
    baseline->executable = strdup(executable ? executable : "");
    baseline->baseline_established = time(NULL);
    baseline->network.tcp_capacity = 16;
    baseline->network.udp_capacity = 16;
    baseline->network.tcp_connections = calloc(baseline->network.tcp_capacity, sizeof(NetworkBaseline));
    baseline->network.udp_connections = calloc(baseline->network.udp_capacity, sizeof(NetworkBaseline));
    baseline->files.capacity = 16;
    baseline->files.patterns = calloc(baseline->files.capacity, sizeof(char*));
    baseline->resources.sample_count = 0;

    store->count++;
    return baseline;
}

int baseline_load(BaselineStore *store, const char *baseline_dir) {
    if (!store || !baseline_dir) {
        return -1;
    }

    DIR *dir = opendir(baseline_dir);
    if (!dir) {
        return 0; // Directory doesn't exist yet, not an error
    }

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (entry->d_name[0] == '.') {
            continue;
        }

        char path[MAX_PATH_LEN];
        snprintf(path, sizeof(path), "%s/%s", baseline_dir, entry->d_name);

        json_t *json = json_utils_load_file(path);
        if (!json) {
            continue;
        }

        ProcessBaseline *baseline = calloc(1, sizeof(ProcessBaseline));
        if (!baseline) {
            json_decref(json);
            continue;
        }

        json_utils_get_int_value(json, "pid", (int*)&baseline->pid);
        json_utils_get_string_value(json, "name", &baseline->name);
        json_utils_get_string_value(json, "executable", &baseline->executable);
        
        const char *timestamp_str;
        json_t *ts = json_utils_get_string_safe(json, "baseline_established");
        if (ts) {
            timestamp_str = json_string_value(ts);
            // Simple parsing (would use proper ISO8601 parser in production)
            baseline->baseline_established = time(NULL);
        }

        // Load network baselines
        json_t *network = json_utils_get_object_safe(json, "network");
        if (network) {
            json_t *tcp = json_utils_get_array_safe(network, "tcp_connections");
            if (tcp) {
                size_t array_size = json_array_size(tcp);
                if (array_size > baseline->network.tcp_capacity) {
                    baseline->network.tcp_capacity = array_size;
                    baseline->network.tcp_connections = realloc(baseline->network.tcp_connections,
                                                               baseline->network.tcp_capacity * sizeof(NetworkBaseline));
                }
                
                size_t index;
                json_t *value;
                json_array_foreach(tcp, index, value) {
                    char *ip;
                    int port;
                    double freq;
                    if (json_utils_get_string_value(value, "dest_ip", &ip) &&
                        json_utils_get_int_value(value, "dest_port", &port) &&
                        json_utils_get_double_value(value, "frequency", &freq)) {
                        
                        NetworkBaseline *nb = &baseline->network.tcp_connections[baseline->network.tcp_count];
                        nb->dest_ip = ip;
                        nb->dest_port = (uint16_t)port;
                        nb->frequency = freq;
                        nb->count = 0;
                        baseline->network.tcp_count++;
                    }
                }
            }
        }

        // Load resource baselines
        json_t *resources = json_utils_get_object_safe(json, "resources");
        if (resources) {
            json_utils_get_double_value(resources, "cpu_mean", &baseline->resources.cpu_mean);
            json_utils_get_double_value(resources, "cpu_std", &baseline->resources.cpu_std);
            json_utils_get_double_value(resources, "memory_mean", &baseline->resources.memory_mean);
            json_utils_get_double_value(resources, "memory_std", &baseline->resources.memory_std);
        }

        // Add to store
        if (store->count >= store->capacity) {
            store->capacity *= 2;
            store->baselines = realloc(store->baselines, store->capacity * sizeof(ProcessBaseline));
        }
        store->baselines[store->count++] = *baseline;
        free(baseline);

        json_decref(json);
    }

    closedir(dir);
    return 0;
}

int baseline_save(BaselineStore *store, const char *baseline_dir) {
    if (!store || !baseline_dir) {
        return -1;
    }

    // Create directory if it doesn't exist
    char cmd[MAX_PATH_LEN + 20];
    snprintf(cmd, sizeof(cmd), "mkdir -p %s", baseline_dir);
    system(cmd);

    for (size_t i = 0; i < store->count; i++) {
        ProcessBaseline *baseline = &store->baselines[i];
        
        json_t *json = json_object();
        json_object_set_new(json, "pid", json_integer(baseline->pid));
        json_object_set_new(json, "name", json_string(baseline->name));
        json_object_set_new(json, "executable", json_string(baseline->executable));
        
        char timestamp[64];
        strftime(timestamp, sizeof(timestamp), "%Y-%m-%dT%H:%M:%SZ", gmtime(&baseline->baseline_established));
        json_object_set_new(json, "baseline_established", json_string(timestamp));

        // Network baselines
        json_t *network = json_object();
        json_t *tcp_array = json_array();
        for (size_t j = 0; j < baseline->network.tcp_count; j++) {
            NetworkBaseline *nb = &baseline->network.tcp_connections[j];
            json_t *conn = json_object();
            json_object_set_new(conn, "dest_ip", json_string(nb->dest_ip));
            json_object_set_new(conn, "dest_port", json_integer(nb->dest_port));
            json_object_set_new(conn, "frequency", json_real(nb->frequency));
            json_array_append_new(tcp_array, conn);
        }
        json_object_set_new(network, "tcp_connections", tcp_array);
        json_object_set_new(json, "network", network);

        // Resource baselines
        json_t *resources = json_object();
        json_object_set_new(resources, "cpu_mean", json_real(baseline->resources.cpu_mean));
        json_object_set_new(resources, "cpu_std", json_real(baseline->resources.cpu_std));
        json_object_set_new(resources, "memory_mean", json_real(baseline->resources.memory_mean));
        json_object_set_new(resources, "memory_std", json_real(baseline->resources.memory_std));
        json_object_set_new(json, "resources", resources);

        // File patterns
        json_t *file_patterns = json_array();
        for (size_t j = 0; j < baseline->files.count; j++) {
            json_array_append_new(file_patterns, json_string(baseline->files.patterns[j]));
        }
        json_object_set_new(json, "file_patterns", file_patterns);

        // Save to file
        char filename[MAX_PATH_LEN];
        snprintf(filename, sizeof(filename), "%s/%s_%d.json", baseline_dir, baseline->name, baseline->pid);
        json_utils_save_file(filename, json);
        json_decref(json);
    }

    return 0;
}

ProcessBaseline* baseline_find(BaselineStore *store, pid_t pid) {
    if (!store) {
        return NULL;
    }

    for (size_t i = 0; i < store->count; i++) {
        if (store->baselines[i].pid == pid) {
            return &store->baselines[i];
        }
    }
    return NULL;
}

ProcessBaseline* baseline_find_by_name(BaselineStore *store, const char *name, const char *executable) {
    if (!store || !name) {
        return NULL;
    }

    for (size_t i = 0; i < store->count; i++) {
        if (strcmp(store->baselines[i].name, name) == 0) {
            if (!executable || strcmp(store->baselines[i].executable, executable) == 0) {
                return &store->baselines[i];
            }
        }
    }
    return NULL;
}

int baseline_add_observation(ProcessBaseline *baseline, BehaviorSnapshot *snapshot) {
    if (!baseline || !snapshot) {
        return -1;
    }

    // Add network observations
    for (size_t i = 0; i < snapshot->network.count; i++) {
        NetworkConnection *conn = &snapshot->network.connections[i];
        network_baseline_add_connection(&baseline->network, conn->dest_ip, conn->dest_port, conn->protocol);
    }

    // Add file observations
    for (size_t i = 0; i < snapshot->files.count; i++) {
        file_baseline_add_pattern(&baseline->files, snapshot->files.files[i]);
    }

    // Update resource statistics (Welford's algorithm for online mean/std)
    baseline->resources.sample_count++;
    uint64_t n = baseline->resources.sample_count;
    double delta_mem = (double)snapshot->resources.memory_rss - baseline->resources.memory_mean;
    baseline->resources.memory_mean += delta_mem / n;
    double delta2_mem = (double)snapshot->resources.memory_rss - baseline->resources.memory_mean;
    baseline->resources.memory_std = sqrt((baseline->resources.memory_std * baseline->resources.memory_std * (n - 1) + delta_mem * delta2_mem) / n);

    return 0;
}

int baseline_finalize(ProcessBaseline *baseline) {
    if (!baseline) {
        return -1;
    }

    // Calculate frequencies for network connections
    uint64_t total_tcp = 0;
    for (size_t i = 0; i < baseline->network.tcp_count; i++) {
        total_tcp += baseline->network.tcp_connections[i].count;
    }
    for (size_t i = 0; i < baseline->network.tcp_count; i++) {
        if (total_tcp > 0) {
            baseline->network.tcp_connections[i].frequency = (double)baseline->network.tcp_connections[i].count / total_tcp;
        }
    }

    uint64_t total_udp = 0;
    for (size_t i = 0; i < baseline->network.udp_count; i++) {
        total_udp += baseline->network.udp_connections[i].count;
    }
    for (size_t i = 0; i < baseline->network.udp_count; i++) {
        if (total_udp > 0) {
            baseline->network.udp_connections[i].frequency = (double)baseline->network.udp_connections[i].count / total_udp;
        }
    }

    return 0;
}

void process_baseline_free(ProcessBaseline *baseline) {
    if (!baseline) {
        return;
    }

    free(baseline->name);
    free(baseline->executable);

    // Free network baselines
    for (size_t i = 0; i < baseline->network.tcp_count; i++) {
        free(baseline->network.tcp_connections[i].dest_ip);
    }
    free(baseline->network.tcp_connections);

    for (size_t i = 0; i < baseline->network.udp_count; i++) {
        free(baseline->network.udp_connections[i].dest_ip);
    }
    free(baseline->network.udp_connections);

    // Free file patterns
    for (size_t i = 0; i < baseline->files.count; i++) {
        free(baseline->files.patterns[i]);
    }
    free(baseline->files.patterns);
}

