#include "config.h"
#include "json_utils.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/stat.h>

#define DEFAULT_LEARNING_PERIOD 86400  // 24 hours
#define DEFAULT_MONITORING_INTERVAL 5  // 5 seconds
#define DEFAULT_RESOURCE_THRESHOLD 3.0 // 3 standard deviations

static char* get_home_dir(void) {
    const char *home = getenv("HOME");
    if (home) {
        return strdup(home);
    }

    struct passwd *pw = getpwuid(getuid());
    if (pw && pw->pw_dir) {
        return strdup(pw->pw_dir);
    }

    return NULL;
}

Config* config_default(void) {
    Config *config = calloc(1, sizeof(Config));
    if (!config) {
        return NULL;
    }

    char *home = get_home_dir();
    if (home) {
        char path[1024];
        snprintf(path, sizeof(path), "%s/.pwarden/config.json", home);
        config->config_path = strdup(path);
        snprintf(path, sizeof(path), "%s/.pwarden/baselines", home);
        config->baseline_dir = strdup(path);
        snprintf(path, sizeof(path), "%s/.pwarden/pwarden.log", home);
        config->logfile_path = strdup(path);
        free(home);
    } else {
        config->config_path = strdup("/etc/pwarden.conf");
        config->baseline_dir = strdup("/var/lib/pwarden/baselines");
        config->logfile_path = strdup("/var/log/pwarden.log");
    }

    config->learning_period_seconds = DEFAULT_LEARNING_PERIOD;
    config->monitoring_interval_seconds = DEFAULT_MONITORING_INTERVAL;
    config->resource_threshold = DEFAULT_RESOURCE_THRESHOLD;
    config->excluded_processes = NULL;
    config->excluded_count = 0;
    config->stdout_enabled = true;
    config->logfile_enabled = true;
    config->syslog_enabled = true;
    config->daemonize = false;

    return config;
}

Config* config_load(const char *config_path) {
    Config *config = config_default();
    if (!config) {
        return NULL;
    }

    // Override default path if provided
    if (config_path) {
        free(config->config_path);
        config->config_path = strdup(config_path);
    }

    // Try to load config file
    json_t *json = json_utils_load_file(config->config_path);
    if (!json) {
        // Config file doesn't exist, return defaults
        return config;
    }

    // Load configuration values
    json_utils_get_string_value(json, "baseline_dir", &config->baseline_dir);
    json_utils_get_string_value(json, "logfile_path", &config->logfile_path);

    int learning_period;
    if (json_utils_get_int_value(json, "learning_period_seconds", &learning_period)) {
        config->learning_period_seconds = (time_t)learning_period;
    }

    int monitoring_interval;
    if (json_utils_get_int_value(json, "monitoring_interval_seconds", &monitoring_interval)) {
        config->monitoring_interval_seconds = monitoring_interval;
    }

    json_utils_get_double_value(json, "resource_threshold", &config->resource_threshold);

    // Load excluded processes
    json_t *excluded = json_utils_get_array_safe(json, "excluded_processes");
    if (excluded) {
        size_t index;
        json_t *value;
        config->excluded_count = json_array_size(excluded);
        config->excluded_processes = calloc(config->excluded_count, sizeof(char*));
        
        json_array_foreach(excluded, index, value) {
            if (json_is_string(value)) {
                config->excluded_processes[index] = strdup(json_string_value(value));
            }
        }
    }

    // Load alert settings
    json_t *alerts = json_utils_get_object_safe(json, "alerts");
    if (alerts) {
        json_t *stdout_val = json_object_get(alerts, "stdout");
        if (json_is_boolean(stdout_val)) {
            config->stdout_enabled = json_is_true(stdout_val);
        }

        json_t *logfile_val = json_object_get(alerts, "logfile");
        if (json_is_boolean(logfile_val)) {
            config->logfile_enabled = json_is_true(logfile_val);
        }

        json_t *syslog_val = json_object_get(alerts, "syslog");
        if (json_is_boolean(syslog_val)) {
            config->syslog_enabled = json_is_true(syslog_val);
        }
    }

    json_t *daemonize_val = json_object_get(json, "daemonize");
    if (json_is_boolean(daemonize_val)) {
        config->daemonize = json_is_true(daemonize_val);
    }

    json_decref(json);
    return config;
}

void config_free(Config *config) {
    if (!config) {
        return;
    }

    free(config->config_path);
    free(config->baseline_dir);
    free(config->logfile_path);

    if (config->excluded_processes) {
        for (size_t i = 0; i < config->excluded_count; i++) {
            free(config->excluded_processes[i]);
        }
        free(config->excluded_processes);
    }

    free(config);
}

bool config_is_excluded(Config *config, const char *process_name) {
    if (!config || !process_name) {
        return false;
    }

    for (size_t i = 0; i < config->excluded_count; i++) {
        if (config->excluded_processes[i] && 
            strcmp(config->excluded_processes[i], process_name) == 0) {
            return true;
        }
    }

    return false;
}

