#include "json_utils.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

json_t* json_utils_load_file(const char *filename) {
    FILE *file = fopen(filename, "r");
    if (!file) {
        return NULL;
    }

    json_error_t error;
    json_t *json = json_loadf(file, 0, &error);
    fclose(file);

    return json;
}

int json_utils_save_file(const char *filename, json_t *json) {
    FILE *file = fopen(filename, "w");
    if (!file) {
        return -1;
    }

    int ret = json_dumpf(json, file, JSON_INDENT(2) | JSON_ENSURE_ASCII);
    fclose(file);

    return ret;
}

json_t* json_utils_get_string_safe(json_t *json, const char *key) {
    json_t *value = json_object_get(json, key);
    if (value && json_is_string(value)) {
        return value;
    }
    return NULL;
}

json_t* json_utils_get_number_safe(json_t *json, const char *key) {
    json_t *value = json_object_get(json, key);
    if (value && json_is_number(value)) {
        return value;
    }
    return NULL;
}

json_t* json_utils_get_array_safe(json_t *json, const char *key) {
    json_t *value = json_object_get(json, key);
    if (value && json_is_array(value)) {
        return value;
    }
    return NULL;
}

json_t* json_utils_get_object_safe(json_t *json, const char *key) {
    json_t *value = json_object_get(json, key);
    if (value && json_is_object(value)) {
        return value;
    }
    return NULL;
}

bool json_utils_get_string_value(json_t *json, const char *key, char **value) {
    json_t *str = json_utils_get_string_safe(json, key);
    if (str) {
        const char *str_value = json_string_value(str);
        if (str_value) {
            *value = strdup(str_value);
            return true;
        }
    }
    return false;
}

bool json_utils_get_int_value(json_t *json, const char *key, int *value) {
    json_t *num = json_utils_get_number_safe(json, key);
    if (num) {
        *value = (int)json_integer_value(num);
        return true;
    }
    return false;
}

bool json_utils_get_double_value(json_t *json, const char *key, double *value) {
    json_t *num = json_utils_get_number_safe(json, key);
    if (num) {
        *value = json_real_value(num);
        return true;
    }
    return false;
}

