#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "process_tracker.h"
#include "behavior_collector.h"
#include "baseline.h"
#include "anomaly_detector.h"
#include "alert.h"
#include "config.h"

static volatile sig_atomic_t g_running = 1;
static volatile sig_atomic_t g_reload = 0;

static void signal_handler(int sig) {
    if (sig == SIGTERM || sig == SIGINT) {
        g_running = 0;
    } else if (sig == SIGHUP) {
        g_reload = 1;
    }
}

static int daemonize(void) {
    pid_t pid = fork();
    if (pid < 0) {
        return -1;
    }

    if (pid > 0) {
        exit(0); // Parent exits
    }

    // Child becomes session leader
    if (setsid() < 0) {
        return -1;
    }

    // Fork again to ensure we're not a session leader
    pid = fork();
    if (pid < 0) {
        return -1;
    }

    if (pid > 0) {
        exit(0); // Parent exits
    }

    // Change working directory
    chdir("/");

    // Close file descriptors
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);

    // Redirect to /dev/null
    open("/dev/null", O_RDONLY);
    open("/dev/null", O_WRONLY);
    open("/dev/null", O_WRONLY);

    return 0;
}

static void ensure_directories(Config *config) {
    char cmd[1024];
    
    // Create baseline directory
    snprintf(cmd, sizeof(cmd), "mkdir -p %s", config->baseline_dir);
    system(cmd);

    // Create log directory
    char *logdir = strdup(config->logfile_path);
    char *last_slash = strrchr(logdir, '/');
    if (last_slash) {
        *last_slash = '\0';
        snprintf(cmd, sizeof(cmd), "mkdir -p %s", logdir);
        system(cmd);
    }
    free(logdir);
}

int main(int argc, char *argv[]) {
    Config *config = config_load(NULL);
    if (!config) {
        fprintf(stderr, "Failed to load configuration\n");
        return 1;
    }

    // Parse command line arguments
    int opt;
    while ((opt = getopt(argc, argv, "d")) != -1) {
        switch (opt) {
            case 'd':
                config->daemonize = true;
                break;
            default:
                fprintf(stderr, "Usage: %s [-d]\n", argv[0]);
                fprintf(stderr, "  -d  Run as daemon\n");
                config_free(config);
                return 1;
        }
    }

    // Daemonize if requested
    if (config->daemonize) {
        if (daemonize() < 0) {
            fprintf(stderr, "Failed to daemonize\n");
            config_free(config);
            return 1;
        }
    }

    // Set up signal handlers
    signal(SIGTERM, signal_handler);
    signal(SIGINT, signal_handler);
    signal(SIGHUP, signal_handler);

    // Ensure directories exist
    ensure_directories(config);

    // Initialize subsystems
    ProcessList *process_list = process_tracker_init();
    if (!process_list) {
        fprintf(stderr, "Failed to initialize process tracker\n");
        config_free(config);
        return 1;
    }

    BaselineStore *baseline_store = baseline_init();
    if (!baseline_store) {
        fprintf(stderr, "Failed to initialize baseline store\n");
        process_tracker_destroy(process_list);
        config_free(config);
        return 1;
    }

    // Load existing baselines
    baseline_load(baseline_store, config->baseline_dir);

    // Initialize alert system
    AlertConfig alert_config = {
        .stdout_enabled = config->stdout_enabled,
        .logfile_enabled = config->logfile_enabled,
        .syslog_enabled = config->syslog_enabled,
        .logfile_path = config->logfile_path
    };
    alert_init(&alert_config);

    // Determine if we're in learning mode
    time_t start_time = time(NULL);
    bool learning_mode = true;

    printf("pWarden starting...\n");
    if (learning_mode) {
        printf("Learning mode: collecting baselines for %ld seconds\n", 
               config->learning_period_seconds);
    } else {
        printf("Monitoring mode: watching for anomalies\n");
    }

    // Main loop
    while (g_running) {
        // Check if we should switch from learning to monitoring
        time_t current_time = time(NULL);
        if (learning_mode && (current_time - start_time) >= config->learning_period_seconds) {
            printf("Learning period complete. Finalizing baselines...\n");
            
            // Finalize all baselines
            for (size_t i = 0; i < baseline_store->count; i++) {
                baseline_finalize(&baseline_store->baselines[i]);
            }
            
            // Save baselines
            baseline_save(baseline_store, config->baseline_dir);
            learning_mode = false;
            printf("Switching to monitoring mode\n");
        }

        // Reload configuration if requested
        if (g_reload) {
            g_reload = 0;
            Config *new_config = config_load(config->config_path);
            if (new_config) {
                config_free(config);
                config = new_config;
                alert_config.stdout_enabled = config->stdout_enabled;
                alert_config.logfile_enabled = config->logfile_enabled;
                alert_config.syslog_enabled = config->syslog_enabled;
                alert_config.logfile_path = config->logfile_path;
                alert_init(&alert_config);
            }
        }

        // Scan for processes
        process_tracker_scan(process_list);

        // Process each tracked process
        ProcessNode *node = process_list->head;
        while (node && g_running) {
            ProcessInfo *process = node->process;
            node = node->next;

            if (!process) {
                continue;
            }

            // Skip excluded processes
            if (config_is_excluded(config, process->name)) {
                continue;
            }

            // Collect behavior snapshot
            BehaviorSnapshot *snapshot = behavior_collector_collect(process->pid);
            if (!snapshot) {
                continue;
            }

            // Find or create baseline
            ProcessBaseline *baseline = baseline_find(baseline_store, process->pid);
            
            if (!baseline) {
                // Try to find by name/executable
                baseline = baseline_find_by_name(baseline_store, process->name, process->executable);
            }

            if (learning_mode) {
                // Learning mode: add observation to baseline
                if (!baseline) {
                    baseline = baseline_create_new(baseline_store, process->pid, 
                                                  process->name, process->executable);
                }
                
                if (baseline) {
                    baseline_add_observation(baseline, snapshot);
                }
            } else {
                // Monitoring mode: check for anomalies
                if (baseline) {
                    // Check each type of anomaly
                    Anomaly *network_anomalies = NULL;
                    size_t network_count = 0;
                    anomaly_detector_check_network(baseline, snapshot, &network_anomalies, &network_count);
                    
                    for (size_t i = 0; i < network_count; i++) {
                        alert_send(&network_anomalies[i]);
                    }
                    
                    if (network_anomalies) {
                        for (size_t i = 0; i < network_count; i++) {
                            free(network_anomalies[i].process_name);
                            free(network_anomalies[i].details);
                        }
                        free(network_anomalies);
                    }
                    
                    Anomaly *resource_anomalies = NULL;
                    size_t resource_count = 0;
                    anomaly_detector_check_resources(baseline, snapshot, config->resource_threshold, 
                                                    &resource_anomalies, &resource_count);
                    
                    for (size_t i = 0; i < resource_count; i++) {
                        alert_send(&resource_anomalies[i]);
                    }
                    
                    if (resource_anomalies) {
                        for (size_t i = 0; i < resource_count; i++) {
                            free(resource_anomalies[i].process_name);
                            free(resource_anomalies[i].details);
                        }
                        free(resource_anomalies);
                    }
                    
                    Anomaly *file_anomalies = NULL;
                    size_t file_count = 0;
                    anomaly_detector_check_files(baseline, snapshot, &file_anomalies, &file_count);
                    
                    for (size_t i = 0; i < file_count; i++) {
                        alert_send(&file_anomalies[i]);
                    }
                    
                    if (file_anomalies) {
                        for (size_t i = 0; i < file_count; i++) {
                            free(file_anomalies[i].process_name);
                            free(file_anomalies[i].details);
                        }
                        free(file_anomalies);
                    }
                } else {
                    // New process without baseline - could be an anomaly
                    // For now, we'll create a baseline and start learning
                    baseline = baseline_create_new(baseline_store, process->pid,
                                                  process->name, process->executable);
                    if (baseline) {
                        baseline_add_observation(baseline, snapshot);
                    }
                }
            }

            behavior_snapshot_free(snapshot);
        }

        // Sleep for monitoring interval
        sleep(config->monitoring_interval_seconds);
    }

    // Cleanup
    printf("pWarden shutting down...\n");
    
    // Save baselines if in learning mode
    if (learning_mode) {
        for (size_t i = 0; i < baseline_store->count; i++) {
            baseline_finalize(&baseline_store->baselines[i]);
        }
        baseline_save(baseline_store, config->baseline_dir);
    }

    alert_cleanup();
    baseline_destroy(baseline_store);
    process_tracker_destroy(process_list);
    config_free(config);

    return 0;
}

