#include "process_tracker.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>

#define MAX_PATH_LEN 4096
#define MAX_CMDLINE_LEN 8192

static char* read_file_content(const char *path) {
    FILE *file = fopen(path, "r");
    if (!file) {
        return NULL;
    }

    char *content = malloc(MAX_CMDLINE_LEN);
    if (!content) {
        fclose(file);
        return NULL;
    }

    size_t len = fread(content, 1, MAX_CMDLINE_LEN - 1, file);
    content[len] = '\0';
    fclose(file);

    // Remove trailing newline
    if (len > 0 && content[len - 1] == '\n') {
        content[len - 1] = '\0';
    }

    return content;
}

static char* read_cmdline(pid_t pid) {
    char path[MAX_PATH_LEN];
    snprintf(path, sizeof(path), "/proc/%d/cmdline", pid);
    
    FILE *file = fopen(path, "r");
    if (!file) {
        return NULL;
    }

    char *cmdline = malloc(MAX_CMDLINE_LEN);
    if (!cmdline) {
        fclose(file);
        return NULL;
    }

    size_t len = fread(cmdline, 1, MAX_CMDLINE_LEN - 1, file);
    fclose(file);

    if (len == 0) {
        free(cmdline);
        return NULL;
    }

    cmdline[len] = '\0';
    
    // Replace null bytes with spaces for readability
    for (size_t i = 0; i < len; i++) {
        if (cmdline[i] == '\0') {
            cmdline[i] = ' ';
        }
    }

    return cmdline;
}

static char* read_executable(pid_t pid) {
    char path[MAX_PATH_LEN];
    char link_target[MAX_PATH_LEN];
    
    snprintf(path, sizeof(path), "/proc/%d/exe", pid);
    ssize_t len = readlink(path, link_target, sizeof(link_target) - 1);
    
    if (len == -1) {
        return NULL;
    }
    
    link_target[len] = '\0';
    return strdup(link_target);
}

static char* read_process_name(pid_t pid) {
    char path[MAX_PATH_LEN];
    snprintf(path, sizeof(path), "/proc/%d/comm", pid);
    return read_file_content(path);
}

static ProcessInfo* create_process_info(pid_t pid) {
    ProcessInfo *info = calloc(1, sizeof(ProcessInfo));
    if (!info) {
        return NULL;
    }

    info->pid = pid;
    info->name = read_process_name(pid);
    info->executable = read_executable(pid);
    info->cmdline = read_cmdline(pid);
    info->first_seen = time(NULL);
    info->last_seen = info->first_seen;

    // Read PPID from stat
    char stat_path[MAX_PATH_LEN];
    snprintf(stat_path, sizeof(stat_path), "/proc/%d/stat", pid);
    FILE *file = fopen(stat_path, "r");
    if (file) {
        // Format: pid (comm) state ppid ...
        int scanned = fscanf(file, "%*d %*s %*c %d", &info->ppid);
        fclose(file);
        if (scanned != 1) {
            info->ppid = 0;
        }
    }

    return info;
}

ProcessList* process_tracker_init(void) {
    ProcessList *list = calloc(1, sizeof(ProcessList));
    return list;
}

void process_tracker_destroy(ProcessList *list) {
    if (!list) {
        return;
    }

    ProcessNode *node = list->head;
    while (node) {
        ProcessNode *next = node->next;
        if (node->process) {
            process_info_free(node->process);
        }
        free(node);
        node = next;
    }
    free(list);
}

int process_tracker_scan(ProcessList *list) {
    if (!list) {
        return -1;
    }

    DIR *proc_dir = opendir("/proc");
    if (!proc_dir) {
        return -1;
    }

    struct dirent *entry;
    bool *seen = calloc(65536, sizeof(bool)); // Track seen PIDs
    
    // Mark all existing processes as not seen
    ProcessNode *node = list->head;
    while (node) {
        if (node->process && node->process->pid < 65536) {
            seen[node->process->pid] = false;
        }
        node = node->next;
    }

    // Scan /proc for processes
    while ((entry = readdir(proc_dir)) != NULL) {
        if (entry->d_type != DT_DIR) {
            continue;
        }

        pid_t pid = (pid_t)atoi(entry->d_name);
        if (pid <= 0) {
            continue;
        }

        if (pid >= 65536) {
            continue; // Skip very large PIDs
        }

        seen[pid] = true;

        // Check if we already have this process
        ProcessInfo *existing = process_tracker_find(list, pid);
        if (existing) {
            existing->last_seen = time(NULL);
            continue;
        }

        // Create new process info
        ProcessInfo *info = create_process_info(pid);
        if (!info) {
            continue;
        }

        // Add to list
        ProcessNode *new_node = calloc(1, sizeof(ProcessNode));
        if (!new_node) {
            process_info_free(info);
            continue;
        }

        new_node->process = info;
        new_node->next = list->head;
        list->head = new_node;
        list->count++;
    }

    closedir(proc_dir);

    // Remove processes that no longer exist
    ProcessNode **prev = &list->head;
    node = list->head;
    while (node) {
        if (node->process && node->process->pid < 65536 && !seen[node->process->pid]) {
            *prev = node->next;
            process_info_free(node->process);
            free(node);
            node = *prev;
            list->count--;
        } else {
            prev = &node->next;
            node = node->next;
        }
    }

    free(seen);
    return 0;
}

ProcessInfo* process_tracker_find(ProcessList *list, pid_t pid) {
    if (!list) {
        return NULL;
    }

    ProcessNode *node = list->head;
    while (node) {
        if (node->process && node->process->pid == pid) {
            return node->process;
        }
        node = node->next;
    }
    return NULL;
}

void process_tracker_remove(ProcessList *list, pid_t pid) {
    if (!list) {
        return;
    }

    ProcessNode **prev = &list->head;
    ProcessNode *node = list->head;

    while (node) {
        if (node->process && node->process->pid == pid) {
            *prev = node->next;
            process_info_free(node->process);
            free(node);
            list->count--;
            return;
        }
        prev = &node->next;
        node = node->next;
    }
}

void process_info_free(ProcessInfo *info) {
    if (!info) {
        return;
    }

    free(info->name);
    free(info->executable);
    free(info->cmdline);
    free(info);
}

